#include "ship.h"

Ship::Ship() {
  Init();
}

Ship::~Ship() {
}

void Ship::Init() {
  NumMissiles = 4;
  ShipRegenerationTime = 1500; // if dead time taken to regenerate
  RechargeCycle = 600;         // time between shots
  ReloadCycle = 0;					   // time between "clips"
	Acceleration = D3DVECTOR(0,0,0);
	Velocity = D3DVECTOR(0,0,0);
	Position = D3DVECTOR(0,0,0);
	AngularVelocity = 0;
	Orientation = 0;
	Dead = false;
};

bool Ship::FireMissile() {
	// check we have missiles to fire
	if(NumMissiles > 0 && ReloadCycle == 0) {
		NumMissiles--;
		ReloadCycle = 100;
		return true;
	}
	return false;
};

void Ship::Update(DWORD dwTime) {
  Model::Update(dwTime);

  // dampen velocity
  Velocity *= 0.99;
  if(Magnitude(Velocity) < 0.01)
    Velocity = D3DVECTOR(0,0,0);

  if(Dead) {
    this->ShipRegenerationTime -= dwTime;
  }

  // Recharge
	if(NumMissiles < 1) {
		RechargeCycle -= dwTime;
		if(RechargeCycle < 1) {
			RechargeCycle = 600;
			NumMissiles = 5;
		}
	}

  // Reload
	if(ReloadCycle>0)
		ReloadCycle -= dwTime;
  if(ReloadCycle<0)
    ReloadCycle = 0;
};

void Ship::Accelerate(bool accelerate, DWORD dwTime) {
  if (accelerate) {
    // 1600 is a fudge factor to smooth the acceleration
    Acceleration.x -= (sin(PI/180 * Orientation) * dwTime)/25000;
	  Acceleration.y += (cos(PI/180 * Orientation) * dwTime)/25000;

		if (Magnitude(Acceleration) > 0.5) {
			Acceleration = Normalize(Acceleration);
		}
		
		// update ship velocity applying acc proportionally in x,y
		Velocity += Normalize(Acceleration) * Magnitude(Acceleration);

		if(Magnitude(Velocity) > 0.5)
			Velocity = 0.5 * Normalize(Velocity);

	} else {
			Acceleration.x = 0;
			Acceleration.y = 0;
	}
};

void Ship::SetAngularVelocity(bool leftKey, bool rightKey, bool snap) {
  this->AngularVelocity = 0;
  if (leftKey != rightKey) {
    if (rightKey)
      this->AngularVelocity = -0.2;
    else
      this->AngularVelocity = 0.2;
    this->Snap = snap;
  }
};

Mesh* Ship::GetMeshData() {
	static D3DVECTOR points[] = { D3DVECTOR(    0.0, 8.0, 0.0),   // 0
															  D3DVECTOR(   -6.0,-8.0, 0.0),   // 1
															  D3DVECTOR(    0.0,-1.0, 0.0),   // 2
															  D3DVECTOR(    0.0,-1.0, 0.0),   // 3
															  D3DVECTOR(    0.0,-1.0, 0.0),   // 4
															  D3DVECTOR(    6.0,-8.0, 0.0),   // 5

															  D3DVECTOR(    0.0,-14.0,0.0),   // 6
															  D3DVECTOR(   -3.0, -9.0,0.0),   // 7
															  D3DVECTOR(    0.0,-17.0,0.0),   // 8
															  D3DVECTOR(    3.0, -9.0,0.0),   // 9
	};
	static int poly01[] = {0,1,2,3,4,5,0};
	static int poly02[] = {6,7,8,9,6};

	Mesh* shipMesh = new Mesh;

	shipMesh->AddPoints(points, sizeof(points)/sizeof(points[0]));
  shipMesh->AddPoly(poly01, sizeof(poly01)/sizeof(int));
  shipMesh->AddPoly(poly02, sizeof(poly02)/sizeof(int));

	return shipMesh;
};

std::vector<std::vector<POINT>> Ship::Lines2d() {
  // get the points
  std::vector<D3DVECTOR> meshD3D(m_mesh->m_points3d.size());

  // rotate and displace the points
	float tcos = cos(PI/180 * Orientation);
	float tsin = sin(PI/180 * Orientation);
  for(unsigned int i = 0; i < meshD3D.size(); i++) {
		if (abs(Orientation)>45)
			Orientation = Orientation;
		meshD3D[i].x = double(m_mesh->m_points3d[i].x) * cos(PI/180 * Orientation) - double(m_mesh->m_points3d[i].y) * sin(PI/180 * Orientation);
		meshD3D[i].y = double(m_mesh->m_points3d[i].y) * cos(PI/180 * Orientation) + double(m_mesh->m_points3d[i].x) * sin(PI/180 * Orientation);
		meshD3D[i].x += Position.x;
		meshD3D[i].y += Position.y;
  }
        
  // build the meshes
	unsigned int limit;
  if (Magnitude(this->Acceleration) < 0.0001) {
    limit = 1;
  } else {
    limit = 2;
  }

  std::vector<std::vector<POINT>> meshes(limit);

  for(size_t j=0; j < limit; j++) {
    std::vector<POINT> mesh(m_mesh->m_polys[j].size());
		
    for(size_t i=0; i < this->m_mesh->m_polys[j].size(); i++) {
			mesh[i].x = meshD3D[this->m_mesh->m_polys[j][i]].x;
			mesh[i].y = meshD3D[this->m_mesh->m_polys[j][i]].y;
		}
    meshes[j] = mesh;
	}

  return meshes;
}

void Ship::Rotate(DWORD dwTime) {
  if (!Snap) {
    Model::Rotate(dwTime);
  } else {
    // in snap mode
	  // if orientation is a multiple of 45, pause briefly (1/4 sec)
    static DWORD pause = 0;
		if (abs(this->Orientation) % 45 <= 0) {
			if(pause == 0) {
				pause = GetTickCount();
			} else {
				if(GetTickCount()-pause > 250) {
					Model::Rotate(dwTime);
					pause = 0;
				}
			}
		} else {
			Model::Rotate(dwTime);
		}
	}
};

