#include "GameMain.h"

BackBuffer* pBackBuffer;

struct Rank {
  unsigned long Score;
  std::string Name;

	static bool Compare(Rank a, Rank b) {
		return (a.Score > b.Score);
	};
};

std::vector<Rank> topTen(10);

unsigned int Lives = 3;
unsigned long Score = 0;
unsigned long Landmark = 10000;
unsigned int numAsteroidsThisLevel = 4;

enum GameMode {
  Pause,
  Play,
  Over,
  Confirm,
  HighScores,
  Help,
	Edit,
};
GameMode GameState = Over;

std::vector<Model*> models;
std::map<std::string, Mesh*> meshes;
Ship ship;
ShipOutline shipOutline;

unsigned int editEntry=0;
DWORD dwStartTime = GetTickCount();
DWORD dwLastTime = 0;
DWORD dwCurrentTime = GetTickCount() - dwStartTime;
DWORD dwTime = 0;
const DWORD dwDelay = 10;

void AddAsteroids() {
  for (unsigned int i = 0; i < numAsteroidsThisLevel; i++) {
    Asteroid* pAsteroid = new Asteroid(meshes[Asteroid::Name()]);

    int tx = pBackBuffer->Width() / 2;
    int ty = pBackBuffer->Height() / 2;
    int rx = RanDouble(-pBackBuffer->Width() / 2, pBackBuffer->Width() / 2);
    int ry = RanDouble(-pBackBuffer->Height() / 2, pBackBuffer->Height() / 2);
    pAsteroid->Position = D3DVECTOR(rx, ry, 0);

    models.push_back(pAsteroid);
  }
};

void GameInsertNewTopScore(char newChar) {
	if(GameState != GameMode::Edit) 
    return;
	// edit current entry
  static unsigned int i = 0;
  switch (newChar) {
  case 0x08: 
    {  // delete
      if(i < 3)
        topTen[editEntry].Name.at(i) = ' ';
      if(i>0) {
        i--;
        topTen[editEntry].Name.at(i) = '_';
      }
      break;
    }
  case 0x0A: 
    {  // enter
      i = 0;
      GameState = GameMode::HighScores;
      break;
    }
  case 0x0D: 
    {  // enter
      i = 0;
      GameState = GameMode::HighScores;

      // save top ten
      std::ofstream ofs("AsteroidsTop10.txt");
	    std::string line;

      if (!ofs.bad()) {
        for(int i = 0; i < 10; i++) {
          ofs << topTen[i].Name << " " << topTen[i].Score << std::endl;
        }
        ofs.close();
      }
      break;
    }
  default:
    {
      if(newChar >= ' ') {
        if(i < 3) {
          topTen[editEntry].Name.at(i) = newChar;
          i++;
          if(i < 3) 
            topTen[editEntry].Name.at(i) = '_';
        }
      }
    }
  }
};

void GameReset() {
  Lives = 3;
  Score = 0;

	ship.Init();

	std::vector<Model*>::iterator it = models.begin();
  for(it = models.begin(); it != models.end(); it++) {
    std::string sss = typeid(*it).name();
    if(typeid(**it) != typeid(Ship))
      delete *it;
  }
  models.clear();

  // add ship to models vector
  models.push_back(&ship);

  // add asteroids to models
  AddAsteroids();
};

void DrawAllModels() {
  // draw shipoutlines to show number of lives
  for(int i = 0; i < Lives; i++) {
    shipOutline.Position = D3DVECTOR( -Model::PositionLimit.x + (i+1) * 16,
                                       Model::PositionLimit.y - 14,
                                       0);
    pBackBuffer->DrawPolyPolyLine(shipOutline.Lines2d());
  }

  std::vector<Model*>::iterator i;
  for(i = models.begin(); i != models.end(); i++) {
    Model& aModel = **i;
		// if this is the ship and it's destroyed don't show
		//if(aModel.same(ship) && ship.Dead) {
		if(aModel.Dead) {
		} else {
      std::string a = typeid(aModel).name();
      pBackBuffer->DrawPolyPolyLine(aModel.Lines2d());
    }
  }
  RECT rcClient = {50, 25, pBackBuffer->Width(), pBackBuffer->Height()};

	// convert score (int) to string
	std::stringstream ss;
	ss << Score;

	pBackBuffer->DrawMsg(&rcClient,0,ss.str().c_str(),0,0);
}

bool RemoveDeadModels(Model* model) {
	return model->Dead;
};

void ShatterAsteroid(Asteroid& asteroid) {
  double min = -0.06;
  double max = 0.06;

  asteroid.Scale(asteroid.Scale() + 1 );
  asteroid.Velocity = D3DVECTOR(RanDouble(min, max), RanDouble(min, max), 0);

  if (asteroid.Scale() >= 4) {
	  asteroid.Dead = true;
  } else {
    for(int i = 0; i<1;i++) {
      Asteroid* pAsteroid = new Asteroid(meshes[Asteroid::Name()]);    

      pAsteroid->Position = asteroid.Position;
      pAsteroid->Scale(asteroid.Scale());
      models.push_back(pAsteroid);
    }
  }
};

void SetState() {
  // determine paused state
  static bool pressedP = false;
  static bool pressedSpace = false;
  static bool pressedI = false;
  static bool pressedS = false;

  if(GameState == GameMode::Play) {
    if (KEYDOWN(0x50)) { // 0x50 is "P" keycode
      if(!pressedP) {
        GameState = GameMode::Pause;
        pressedP = true;
      }
    } else {
      pressedP = false;
    }
    //if (KEYDOWN(0x50)
  };

  if(GameState == GameMode::Pause) {
    if (KEYDOWN(0x50)) { // 0x50 is "P" keycode
      if(!pressedP) {
        GameState = GameMode::Play;
        pressedP = true;
      }
    } else {
      pressedP = false;
    }
  }
    
  if(GameState == GameMode::Over) {
    // check for
    //    Space - start
    if (KEYDOWN(VK_SPACE)) {
      if(!pressedSpace) {
        pressedSpace = true;
      }
    } else {
      if(pressedSpace) {
        GameState = GameMode::Play;
        GameReset();
      };
      pressedSpace = false;
    };
  }

  if(GameState == GameMode::Over) {
    if (KEYDOWN(0x49)) { // 0x49 is "I" keycode
      if(!pressedI) {
        GameState = GameMode::Help;
        pressedI = true;
      }
    } else {
      pressedI = false;
    }    
  };

  if(GameState == GameMode::Help) {
    if (KEYDOWN(0x49)) { // 0x49 is "I" keycode
      if(!pressedI) {
        GameState = GameMode::Over;
        pressedI = true;
      }
    } else {
      pressedI = false;
    }
  }

  if(GameState == GameMode::Over) {
    if (KEYDOWN(0x53)) { // 0x53 is "S" keycode
      if(!pressedS) {
        GameState = GameMode::HighScores;
        pressedS = true;
      }
    } else {
      pressedS = false;
    }    
  };

  if(GameState == GameMode::HighScores) {
    if (KEYDOWN(0x53)) { // 0x53 is "S" keycode
      if(!pressedS) {
        GameState = GameMode::Over;
        pressedS = true;
      }
    } else {
      pressedS = false;
    }
  }
  /*
  if(KEYDOWN(0x50)) { // 0x50 is "P" keycode
  }
  if(KEYDOWN(0x48)) { // 0x48 is "H" keycode
  }
  if(KEYDOWN(VK_F2)) {
  }
  if(KEYDOWN(0x54)) { // 0x54 is "S" keycode
  }
  if(KEYDOWN(VK_ESC)) {
  }
  if(KEYDOWN(0x51)) { // 0x51 is "Q" keycode
  }
  if(KEYDOWN(VK_TAB)) { // 0x50 is "P" keycode
  }
*/
};

void ConfirmQuit() {
  pBackBuffer->Clear();
  
  DrawAllModels();

  RECT rcClient = {0, pBackBuffer->Height()/20, pBackBuffer->Width(), pBackBuffer->Height()};
	pBackBuffer->DrawMsg(&rcClient,0,"Asteroids",2,DT_CENTER);
	
  rcClient.top = pBackBuffer->Height()*2/3;	
  std::string msg = "Finish game?\r\rPress Y to confirm\rPress N to continue";
  pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_CENTER | DT_VCENTER);

  if(KEYDOWN(0x59))
    GameState = GameMode::Over;
  else if (KEYDOWN(0x4E))
    GameState = GameMode::Play;

  pBackBuffer->Display();
};

void CheckKeyboard() {  
  // if the ship is not dead process any key presses
  if (ship.Dead) 
    return;

  static bool pressedTab = false;

  if (KEYDOWN(0x51)) { // pressed (Q)uit game...
    GameState = GameMode::Confirm;    
    return;
  }

	// if space key pressed fire missiles
	if (KEYDOWN(VK_SPACE)) {
		if(ship.FireMissile()) {
			Missile* pMissile = new Missile(ship);				
      pMissile->setMesh(meshes[Missile::Name()]);
			models.push_back(pMissile);
		}
	}

  // if delete key pressed reset ship position
  if (KEYDOWN(VK_DELETE)) {
    ship.Orientation = 0;
    ship.Position = D3DVECTOR(0,0,0);
    ship.Velocity = D3DVECTOR(0,0,0);
  }

  // if tab key pressed warp ship position
  if (KEYDOWN(VK_TAB)) {
    pressedTab = true;
  } else {
    if(pressedTab) {
      pressedTab = false;
      ship.Orientation = 0;
      
      int i = 0;
      D3DVECTOR newPosition;
      while (i < models.size()) {
        // don't want to warp into an asteroid...
        int rx = RanDouble(-200,200);
        int ry = RanDouble(-200,200);
        newPosition = D3DVECTOR(rx,ry,0);
        for(; i < models.size() ; i++) {
          Model& rModel = *models[i];
          if(typeid(rModel) == typeid(Asteroid)) {
            double distance = Magnitude(newPosition - rModel.Position);
		        if(distance < rModel.BoundingRadius) {
              break;
            }
          }
        }
      }

      ship.Position = newPosition;
      ship.Velocity = D3DVECTOR(0,0,0);
      ship.Acceleration = D3DVECTOR(0,0,0);
    }
  }

  // if insert pressed kill ship
  ship.Dead = KEYDOWN(VK_INSERT);

  // if up pressed accelerate ship
  ship.Accelerate(KEYDOWN(VK_UP), dwTime);

  // process left, right and shift
  ship.SetAngularVelocity(KEYDOWN(VK_LEFT), KEYDOWN(VK_RIGHT), KEYDOWN(VK_SHIFT));
}

void CheckCollisions() {
  /////////////////////////////////////////////////////////////////////////////////
  // collisions
  //          | asteroid | ship    | debris | missile
  // ---------+----------+---------+--------+---------
  // asteroid |  ignore  | shatter | ignore | shatter 
  // ship     |  die     | never   | never  | die
  // debris   |  ignore  | never   | ignore | ignore
  // missile  |  die     | die     | ignore | ignore

  // "never" and "ignore" are practically the same and mean that the comparison can be skipped
  // this requires that the object types are identified (rtti!)
  // results are shatter or die 
  // die requires the object be removed from the vector, can't remove itself

  // check for collision between ship and asteroids
  if(ship.Dead == false) {
	  for(int i =0; i < models.size() ; i++) {
		  Model& rModel = *models[i];

		  if(typeid(rModel) == typeid(Asteroid)) {
			  double distance = Magnitude(ship.Position - rModel.Position);
			  if(distance < ship.BoundingRadius || distance < rModel.BoundingRadius) {
				  // ship hit
          ship.Dead = true;
				  Lives--;
				  for(int i = 0; i < 4; i++) {
					  Debris* pDebris = new Debris(meshes[Debris::Name()]);

					  pDebris->Position = ship.Position;
					  pDebris->Orientation = ship.Orientation;
					  pDebris->Velocity = ship.Velocity ;
					  pDebris->Velocity.x += RanDouble(-0.1,0.1);
					  pDebris->Velocity.y += RanDouble(-0.1,0.1);
					  pDebris->Lifetime = ship.ShipRegenerationTime - 500;

					  models.push_back(pDebris);
				  }	
				  break;
			  }
		  }
	  }
  } else {
	  if(ship.ShipRegenerationTime < 1) {
      // regeneration cycle complete - reinitialise ship
      int i=0;
      // make sure ship won't materialise inside asteroid
      for(; i<models.size(); i++) {
		    Model& rModel = *models[i];

		    if(typeid(rModel) == typeid(Asteroid)) {
			    double distance = Magnitude(rModel.Position) ;
			    if(distance < rModel.BoundingRadius) {
						break;
          }
				}
      }

      // reinitialise ship
			if(i>=models.size()) {
				ship.Init();
			}
		}
  };

  // check for collision between missiles and asteroids
  // can be in any order (missiles and asteroids are added to the end of the vector)
  // when an item is erased it invalidates the index an count of the vector
  // either restart the process after each erase
  // or mark for deletion later

  // either both the missile and ship are deleted
  // or erase missile and shatter asteroid
  // can't for through the models as size changes when erasing or adding items	
  for(int i = 0; i < models.size() ; i++) {
	  Model& rMissile = *models[i];
	  if(typeid(rMissile) == typeid(Missile)) {
		  for(int j = 0; j < models.size() ; j++) {
			  if(typeid(*models[j]) == typeid(Asteroid)) {
				  Asteroid& rAsteroid = static_cast<Asteroid&>(*models[j]);

          double distance = Magnitude(rAsteroid.Position - rMissile.Position);
				  if(distance < rAsteroid.BoundingRadius || distance < rMissile.BoundingRadius) {
					  // missile hit asteroid            
            // if score passes 10k, 20k etc landmark get extra life...
            Score += rAsteroid.Scale() * 100;
            if(Score >= Landmark) {
              ++Lives;
              Landmark += 10000;
            }
            rMissile.Dead = true;
            ShatterAsteroid(rAsteroid);
					  break;
				  }
			  }
		  }
	  }
  }

  // add more asteroids if required
  if(Asteroid::numAsteroids == 0) {
    if(numAsteroidsThisLevel < 10) {
      ++numAsteroidsThisLevel;
    }
    AddAsteroids(); 
  }
};

void GameInit(){
	// set up back buffer
  if(pBackBuffer==NULL)	{
    pBackBuffer = new BackBuffer;
	  pBackBuffer->Create();
  };

  // set up default top 10
  topTen[0].Name = "DCB";
  topTen[1].Name = "ADM";
  topTen[2].Name = "ABC";
  topTen[3].Name = "DEF";
  topTen[4].Name = "GHI";
  topTen[5].Name = "JKL";
  topTen[6].Name = "MNO";
  topTen[7].Name = "QRS";
  topTen[8].Name = "TUV";
  topTen[9].Name = "WXY";

  for(int i = 0;i<10;i++) {
    topTen[i].Score = (9-i)*10000;
  }

  // read from top 10 file if exists
  std::ifstream ifs("AsteroidsTop10.txt");
	std::string line;

  if (ifs != NULL && !ifs.bad()) {
		while (! ifs.eof() ) {
			getline (ifs,line);
			
      if(line.length() > 5) {
        Rank a;
			  a.Name = line.substr(0,3);
			  StringToNumber( line.substr(4), a.Score);
			  topTen.push_back(a);
      }
		}
    ifs.close();

		sort(topTen.begin(), topTen.end(), Rank::Compare);
		topTen.erase(topTen.begin()+10, topTen.end());
  }

  Model::PositionLimit.x = pBackBuffer->Width() / 2;
  Model::PositionLimit.y = pBackBuffer->Height() / 2;

  /////////////////////////////////////////////////////////////////////////////////
  // add model meshes to meshes map
  // add ship mesh
	meshes[Ship::Name()] = Ship::GetMeshData();
  // add missile mesh
	meshes[Missile::Name()] = Missile::GetMeshData();
  // add asteroid mesh
	meshes[Asteroid::Name()] = Asteroid::GetMeshData();
  // add debris mesh
	meshes[Debris::Name()] = Debris::GetMeshData();

  /////////////////////////////////////////////////////////////////////////////////
  // set ship mesh
	ship.setMesh(meshes[Ship::Name()]);

  /////////////////////////////////////////////////////////////////////////////////
  // set shipOutline mesh
	shipOutline.setMesh(meshes[Ship::Name()]);

  GameReset();
};

void GameOver() {
  // update positions
  UpdateAllModels();

  pBackBuffer->Clear();
  
  DrawAllModels();

  RECT rcClient = {0, pBackBuffer->Height()/20, pBackBuffer->Width(), pBackBuffer->Height()};
	pBackBuffer->DrawMsg(&rcClient,0,"Asteroids",2,DT_CENTER);
	
  rcClient.top = pBackBuffer->Height()*1/6;	
  std::string msg = "by";
  msg += "\rCarl Bateman";
	pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_CENTER | DT_VCENTER);

  rcClient.top = pBackBuffer->Height()*2/3;	
  msg = "space to start";
  msg += "\r";
  msg += "\rI for instructions";
  msg += "\r";
  msg += "\rs for high scores";
	pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_CENTER | DT_VCENTER);

  pBackBuffer->Display();
};

void GameShowScores() {
	std::stringstream ss;
  
  UpdateAllModels();

  pBackBuffer->Clear();
  
  DrawAllModels();

  RECT rcClient = {0, pBackBuffer->Height()/20, pBackBuffer->Width(), pBackBuffer->Height()};
	pBackBuffer->DrawMsg(&rcClient,0,"Asteroids",2,DT_CENTER);
	
  rcClient.top = pBackBuffer->Height()/6;
  rcClient.left = 0;
  rcClient.right = pBackBuffer->Width()/2;
  std::string msg = "\r";
  for(int i=0; i<topTen.size();i++) {
    msg += "\r" + NumberToString(i+1) + "  ";
    msg += topTen[i].Name + "  ";
  }

	pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_RIGHT);
	
  rcClient.left = pBackBuffer->Width()/2;
  rcClient.right = pBackBuffer->Width();
  msg = "\r";
  for(int i=0; i<topTen.size();i++) {
    msg += "\r" + NumberToString(topTen[i].Score);
  }
	pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_LEFT);

  rcClient.left = 0;
  rcClient.right = pBackBuffer->Width();
  if(GameState == GameMode::HighScores)
    msg = "High Scores\r\r\r\r\r\r\r\r\r\r\r\r\rpress S to continue";
  else
    msg = "High Scores\r\r\r\r\r\r\r\r\r\r\r\r\rpress enter to finish editing";
	pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_CENTER | DT_LEFT);

  pBackBuffer->Display();
};

void GameShowHelp() {
  UpdateAllModels();

  pBackBuffer->Clear();
  
  DrawAllModels();

  RECT rcClient = {0, pBackBuffer->Height()/20, pBackBuffer->Width(), pBackBuffer->Height()};
	pBackBuffer->DrawMsg(&rcClient,0,"Asteroids",2,DT_CENTER);

  rcClient.top = pBackBuffer->Height()/6;
  rcClient.left = 0;
  rcClient.right = pBackBuffer->Width()/2;
  std::string msg = "\r\r";
  msg += "\rLeft ";
  msg += "\rRight ";
  msg += "\rUp ";
  msg += "\rTab ";
  msg += "\rSpace ";
  msg += "\rQ ";
	pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_RIGHT);
	
  rcClient.left = pBackBuffer->Width()/2;
  rcClient.right = pBackBuffer->Width();
  msg = "\r\r";
  msg += "\r rotate left";
  msg += "\r rotate right";
  msg += "\r accelerate";
  msg += "\r warp";
  msg += "\r fire";
  msg += "\r end game";
	pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_LEFT);

  rcClient.left = 0;
  rcClient.right = pBackBuffer->Width();
  msg = "\rControl Keys\r\r\r\r\r\r\r\r\r\r\rpress I to continue";
	pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_CENTER | DT_LEFT);

  pBackBuffer->Display();
};

void GamePause() {
  pBackBuffer->Clear();

  DrawAllModels();
	
  RECT rcClient = {0, pBackBuffer->Height()/20, pBackBuffer->Width(), pBackBuffer->Height()};
	pBackBuffer->DrawMsg(&rcClient,0,"Asteroids",2,DT_CENTER);
	
  rcClient.top = pBackBuffer->Height()*2/3;	
  std::string msg = "\rPaused\r\r\rPress P to continue";
  pBackBuffer->DrawMsg(&rcClient,0,msg.c_str(),0,DT_WORDBREAK | DT_CENTER | DT_VCENTER);

  pBackBuffer->Display();
};

void GamePlayGame() {
  UpdateAllModels();

  CheckKeyboard();
  CheckCollisions();

	if(Lives < 1) {
		// game over
		if(Score > topTen[9].Score) {
			// new score for top ten
			GameState = GameMode::Edit;
      topTen[9].Name = "_  ";
      topTen[9].Score = Score;
  		sort(topTen.begin(), topTen.end(), Rank::Compare);

      // find the place to insert new score      
      for(editEntry = 0; editEntry<topTen.size(); editEntry++) {
        if (Score == topTen[editEntry].Score && "_  " == topTen[editEntry].Name) {
          break;
        }
      }
		} else {
			GameState = GameMode::Over;
		}
	}

  // delete all models marked as dead
	std::vector<Model*>::iterator fwd = models.begin();
	std::vector<Model*>::iterator bwd = models.end()-1;
	while(fwd != bwd+1) {
    if ((*fwd)->Dead && typeid(**fwd) != typeid(Ship)) {
      delete *fwd;
		  *fwd = *bwd;
		  bwd--;
		} else {
			++fwd;
		};
	}

	models.erase(fwd, models.end());

  pBackBuffer->Clear();

  DrawAllModels();
	
  pBackBuffer->Display();
};

void UpdateAllModels() {
  // update position, orientation, etc of all models
  std::vector<Model*>::iterator i;
  for(i = models.begin(); i != models.end(); i++) {
    // assign to temp var for readability
	  Model& rModel = **i;
    
    rModel.Update(dwTime);
  }
};

void GameMain(){
  dwCurrentTime = GetTickCount() - dwStartTime;
  if((dwCurrentTime - dwLastTime) < dwDelay)
    return;
  // calculate the actual time between frames
  dwTime = (dwCurrentTime - dwLastTime);
  // update time between last two frames
  dwLastTime = dwCurrentTime;
  
	SetState();
  
  if(GameState == Play) {
    GamePlayGame();
  } if(GameState == GameMode::Over) {
    GameOver();
  } if(GameState == GameMode::Help) {
    GameShowHelp();
  } if(GameState == GameMode::Pause) {
    GamePause();
  } if(GameState == GameMode::HighScores) {
    GameShowScores();
  } if(GameState == GameMode::Edit) {
    GameShowScores();
  } if(GameState == GameMode::Confirm) {
    ConfirmQuit();
  }
};

void GameExit(){
  // delete all objects in models vector
  std::vector<Model*>::iterator i;
  for(i = models.begin(); i != models.end(); i++) {
    if(typeid(**i) != typeid(Ship))
      delete *i;
  }

  pBackBuffer->Destroy();
};